#ifndef AHLGREN_SATSOLVER
#define AHLGREN_SATSOLVER

#include <iostream>
#include <vector>
#include <map>
#include <unordered_map>
#include <set>
#include <functional> // greater<double> for VSIDS
#include <memory> // shared_ptr for bit_to_count jump table

#include "program.h"
#include "mode.h"
#include "global.h"

namespace lp {
	using namespace std;

	class Program;

	// Error class for max assignments
	struct sat_exception : public std::exception {};
	struct max_assignments : public sat_exception {};
	// Error class for pop()
	struct unmodifiable : public std::exception {
		unmodifiable() : std::exception() {}
	};

	class PClause;
	struct pvar {
		// Constructor for choice assignment
		pvar() : val(0), ante(nullptr), level(-1), sublevel(0), both_ways(false) {}
		pvar(int v, int d, int sd, bool closed = false) : val(v), ante(nullptr), level(d), sublevel(sd), both_ways(closed) {}
		// Constructor for propagated assignment
		pvar(int v, int d, int sd, const PClause* ptr) : val(v), ante(ptr), level(d), sublevel(sd), both_ways(true) {}
		// Set?
		bool is_set() const { return val != 0; }
		// Decision/propagation point?
		bool is_decision() const { return !ante && is_set(); }
		bool is_propagated() const { return ante && is_set(); }
		// Fixed?
		bool is_fixed() const { return level < 0 && is_set(); }
		// Data members
		int val; // var=1,2,3,4,... (positive or negative)
		const PClause* ante; // antecedent
		int level; // depth level
		int sublevel; // sublevel (higher = later)
		bool both_ways; // tried both ways?
	};

	// *** Define Variable Assignment class *** //
	class Vass {
	public:
		// Typedefs
		typedef std::vector<pvar> var_table;
		typedef std::vector<pvar*> model;
		typedef model::size_type size_type;
		typedef model::iterator iterator;
		typedef model::const_iterator const_iterator;

		// Constructors
		Vass(int max_ones = numeric_limits<int>::max(), int dm = numeric_limits<int>::max())
			: ones(0), m1(max_ones), ac(0), amax(dm) {}
		Vass(const Functor& bot, const vector<Mode>& fm, int max_ones, int dm)
			: ones(0), m1(max_ones), ac(0), amax(dm) { init_model(bot.body_size()); init_modes(bot,fm); }
		Vass(const Vass& v) { *this = v; }
		Vass(Vass&& v) { *this = std::move(v); }
		Vass& operator=(const Vass& v);
		Vass& operator=(Vass&& v);

		// Compare two variable assignments, order does not matter
		//bool operator==(const Vass& vass) const;
		//bool operator!=(const Vass& vass) const { return !(*this == vass); }

		// Initialize modes, variable table, and reserve vass
		void init_modes(const Functor& bot, const vector<Mode>& fm);
		void init_model(int model_size);

		// DEBUG: check that mode counter is ok
		bool check_counter() const {
			int idx = 0;
			for (auto i = bit_to_count.begin(); i != bit_to_count.end(); ++i, ++idx) {
				if ( (*i)->first > (*i)->second ) {
					cerr << "Counter error: " << (*i)->first << " > " << (*i)->second << "\n";
					cerr << "Literal: " << idx << "\n";
					return false;
				}
			}
			return true;
		}

		// Get/Set max number of ones
		int max1() const { return m1; }
		bool set_max1(int max);
		// Get/Set maximal number of assignments
		int maxa() const { return amax; }
		bool set_maxa(int max);

		void clear_ass();
		void clear();
		bool empty() const { return va.empty(); }
		size_type size() const { return va.size(); }
		bool filled1() const { return ones == m1; }

		// Reset vass before DPLL run
		void reset(int model_size);
		// Reset assignment counter
		void reset_counters();
		// Get value of assignment counter
		int acounter() { return ac; }
		// Push back new assignment
		int insert(const pvar& a);
		// Insert without check consistency/redundancy (but still keep track of ones)
		void push(const pvar& a);
		// Pop last value
		void pop();
		// Get last assignment
		pvar& last_ass() { return *va.back(); }
		// Get i:th assignment
		pvar& operator[](int i) { return *va[i]; }
		// Get variable v
		pvar& get_var(int v) { return pvt[std::abs(v)-1]; }
		const pvar& get_var(int v) const { return pvt[std::abs(v)-1]; }
		// Switch branch and close (return 0 iff failure)
		int flip_back();

		// Limits on number of 1's assigned
		int mode_limit(int i) const { return bit_to_count[i]->second; }
		int& mode_limit(int i) { return bit_to_count[i]->second; }

		// Iterators
		const_iterator begin() const { return va.begin(); }
		const_iterator end() const { return va.end(); }

		// Can we assign a?
		bool is_assignable(int a) const;

		void print(ostream& os) const;
	protected:
		// m1 is set by query set(s,n)
		// bit_to_count is set by third argument to modeb
		int ones; // current number of ones
		int m1; // maximal number of ones
		int ac; // assignment counter
		int amax; // maximal number of assignments
		var_table pvt; // table of variables
		model va; // literal assignments
		// Use jumptable for bit_index -> pair<mode_count,recall>*
		std::vector< std::shared_ptr<std::pair<int,int>> > bit_to_count;
	};

	inline ostream& operator<<(ostream& os, const Vass& v) { v.print(os); return os; }

	// *** Define Watched Clause class *** //
	class PClause {
	public:
		typedef set<int>::iterator iterator;
		typedef set<int>::const_iterator const_iterator;
		PClause() : watch(0,0) {}
		PClause(const PClause& c) : lits(c.lits), watch(c.watch) {}
		PClause& operator=(const PClause& c) {
			if (this != &c) { lits = c.lits; watch = c.watch; } 
			return *this;
		}
		// Note: we use standard copy constructor and assignment operator, define lvalue versions too
		PClause(PClause&& c) : lits(move(c.lits)), watch(move(c.watch)) { }
		PClause& operator=(PClause&& c) { if (this != &c) { lits = move(c.lits); watch = move(c.watch); } return *this; }

		PClause(int l) : watch(l,0) { lits.insert(l); }
		PClause(int x, int y) : watch(x,y) { lits.insert(x); lits.insert(y); }
		PClause(const bitstring& bs);

		template <typename Iter> PClause(Iter beg, Iter end) : watch(0,0) {
			lits.insert(beg,end);
			if (beg != end) {
				watch.first = *beg;
				if (++beg != end) watch.second = *beg;
			}
		}


		// Add literal with value
		bool insert(int lit);
		// Erase literal
		bool erase(int l);
		// Erase from iterator
		const_iterator erase(const_iterator i) { return lits.erase(i); }
		// Clear all literals
		void clear() { lits.clear(); watch.first = watch.second = 0; }
		// Is it a unit clause? (returns the literal or 0 if none)
		int unit() const { return (lits.size() == 1) ? *lits.begin() : 0; }
		// Is it the empty clause?
		bool empty() const { return lits.empty(); }
		// Get clause size
		set<int>::size_type size() const { return lits.size(); }

		// Has literal?
		bool has(int l) const { return lits.find(l) != lits.end(); }
		// Get sign of literal (0 = not present)
		int is_function(int l) const;
		// Comparison operators (ignoring mutable watchlists)
		bool operator==(const PClause& c) const { return lits == c.lits; }
		bool operator!=(const PClause& c) const { return !(lits == c.lits); }
		// Iterator Access
		iterator begin() { return lits.begin(); }
		iterator end() { return lits.end(); }
		const_iterator begin() const { return lits.begin(); }
		const_iterator end() const { return lits.end(); }
		// Are we watching two literals?
		bool watching() const { return watch.first != 0 && watch.second != 0; }
		// Get watched literals
		pair<int,int>& watched() { return watch; }
		pair<int,int> watched() const { return watch; }
		int& watched1() { return watch.first; }
		int watched1() const { return watch.first; }
		int& watched2() { return watch.second; }
		int watched2() const { return watch.second; }
		int other_watched(int l) const;
		bool switch_watched(int lo, int ln);

		// Logical
		bool satisfied(const Vass& va) const;

		// Print
		void print(ostream& os) const;
	protected:
		set<int> lits;
		pair<int,int> watch;
	};

	inline ostream& operator<<(ostream& os, const PClause& c) { c.print(os); return os; }
	// Implication operators
	inline bool operator>>(const Vass& v, const PClause& c) { return c.satisfied(v); }
	inline bool operator<<(const PClause& c, const Vass& v) { return c.satisfied(v); }

	// Declare Function Object for DPLL selection
	class DPLL_Selection;

	// *** Define Clause Database class *** //
	class ClauseDB {
	public:
		// Define clause database
		typedef std::list<PClause> db_type;
		typedef db_type::size_type size_type;
		// Define clause iterator
		typedef db_type::iterator iterator;
		typedef db_type::const_iterator const_iterator;
		// Define Watch List Type
		//typedef multimap<int,iterator> wlist_type;
		// Define watch list iterator
		//typedef wlist_type::iterator wliter;

		ClauseDB() : empty_cl(false) { }
		ClauseDB(
			const Functor& bot,
			int bsize,
			const std::vector<Mode>& lits_to_modes,
			Constraints& constr,
			const parameters& par)
			: empty_cl(false), params(par)
		{
			//db.reserve(bsize*2); // 100% more for clause learning
			init_modes(bot,lits_to_modes,constr,params);
		}

		ClauseDB(ClauseDB&& cdb) 
			: empty_cl(cdb.empty_cl), db(std::move(cdb.db)), ass(std::move(cdb.ass)), params(std::move(cdb.params)) {}

		ClauseDB& operator=(ClauseDB&& cdb)
		{
			if (this != &cdb) {
				empty_cl = cdb.empty_cl;
				db = std::move(cdb.db);
				ass = std::move(cdb.ass);
				params = std::move(cdb.params);
			}
			return *this;
		}

		// Initialize ClauseDB with mode restrictions
		void init_modes(const Functor& bottom, const vector<Mode>& lits_to_modes, Constraints&, const parameters&);

		// Number of clauses
		size_type size() const { return has_empty() + ass.size() + db.size(); }
		// Is empty?
		bool is_empty() const { return !has_empty() && ass.empty() && db.empty(); }
		// Clear
		void clear() { empty_cl = false; ass.clear(); db.clear(); }

		// Do we have the empty clause?
		bool has_empty() const { return empty_cl; }

		// Add PClause to DB if consistent, returns false otherwise
		void insert(PClause&&);
		// Erase clause
		iterator erase(iterator i) { return db.erase(i); }
		// Use DPLL to retrieve model, or return false is unsatisfiable
		std::multimap<int,PClause*> make_lit2cl();
		bool simplify(const Vass& settings, unsigned model_size);
		int dual_horn_solve(Vass& vass, unsigned model_size);
		int dpll(Vass& model, unsigned model_size, DPLL_Selection&, deadline_t deadline, const std::vector<pvar>* suggestions = nullptr);
		int unit_propagate(int level, Vass& model, std::pair<const PClause*,int>* err, const std::multimap<int,PClause*>& lit2cl, 
			deadline_t deadline = deadline_t::max());
		int backtrack(int level, Vass& vass, DPLL_Selection& select, std::pair<const PClause*,int>* err);

		// Traverse DB
		iterator begin() { return db.begin(); }
		iterator end() { return db.end(); }
		const_iterator begin() const { return db.begin(); }
		const_iterator end() const { return db.end(); }

		// Logical
		bool satisfied(const Vass& va) const;

		// Print database
		void print(ostream&) const;
	protected:
		bool empty_cl;
		db_type db; // database does not store unit clauses
		std::vector<int> ass; // this one does
		parameters params;
	private:
		ClauseDB(const ClauseDB& cdb);
		ClauseDB& operator=(const ClauseDB& cdb);
	};

	inline ostream& operator<<(ostream& os, const ClauseDB& cdb) { cdb.print(os); return os; }
	// Implication operators
	inline bool operator>>(const Vass& v, const ClauseDB& db) { return db.satisfied(v); }
	inline bool operator<<(const ClauseDB& db, const Vass& v) { return db.satisfied(v); }


	// ************ Define Helper Functions ****************** //


	// Make functor from bit string and bottom clause
	//term* make_functor_ptr(const term* bot, const bitstring& bs);
	// Make functor with masked "_#" arguments for zeros in bitstring (third argument is dummy)
	//term* make_functor_ptr(const term* bot, const bitstring& bs, bool include_masks);
	// Non-pointer variants
	//std::unique_ptr<Functor> make_functor(const term* bot, const bitstring& bs, bool include_masks);
	//std::unique_ptr<Functor> make_functor(const term* bot, const bitstring& bs);


	// Convert from model to bit string
	bitstring make_bitstring(const Vass& m);
	// Convert functor and bottom clause to bitstring
	//bitstring make_bitstring(const term* bot, const term* cand);

	// Make subspace restriction after pruning functor space
	PClause make_complement_clause(const Vass& va, int consistent);

	// Make mode restrictions based on bottom clause
	//void make_mode_clause(const term* f, const map<std::unique_ptr<Functor>,Mode >& fm, ClauseDB& db);

	//============================ DPLL Selection functions ==========================//

	// Define DPLL Selection Mechanism as Function Object
	class DPLL_Selection {
	public:
		virtual void init(const ClauseDB& db, const Vass& va, int msize) { model_size = msize; }
		virtual int choose(const ClauseDB&, const Vass& va) = 0;
		virtual void update(const ClauseDB& db, const Vass& va, const PClause& constraint) { }
	protected:
		int model_size;
	};

	struct Random_selection : DPLL_Selection {
		virtual int choose(const ClauseDB&, const Vass& va)
		{
			// Note: to get a correct uniform distribution, we need to consider only the positves
			// that are assignable
			std::set<int> uvars;
			// Add all assignable
			for (int k = -model_size; k <= model_size; ++k) {
				if (k != 0 && va.is_assignable(k)) uvars.insert(k);
			}
			// Now filter off variables that have been assigned
			std::for_each(va.begin(),va.end(),[&](const pvar* a){ uvars.erase(a->val); uvars.erase(-a->val); });
			// uvars now contains all unassigned variables that are assignable
			assert(!uvars.empty());
			const auto k = useful::uniform_discrete<decltype(uvars.size())>(0,uvars.size()-1);
			auto i = uvars.begin();
			std::advance(i,k);
			return *i;
		}
	};

	struct Simplicity_selection : DPLL_Selection {
		virtual int choose(const ClauseDB&, const Vass& va)
		{
			// TODO: use datastructure to keep assigned vars
			std::set<int> uvars;
			// Add all assignable negatives
			for (int k = -model_size; k < 0; ++k) {
				if (va.is_assignable(k)) uvars.insert(k);
			}
			// Now filter off negatives that have been assigned
			for_each(va.begin(),va.end(),[&](const pvar* a){uvars.erase(-std::abs(a->val));});
			// uvars now contains all unassigned variables that are assignable
			assert(!uvars.empty());
			// Set -1 first, then -2, then -3, ... Upon backtrack, we thus flip -3 before -2, before -1
			return *(--uvars.end());
		}
	};

	class VSIDS_selection : public DPLL_Selection {
	public:
		typedef map<int,double> counter_type;
		VSIDS_selection(double s) : scale(s) {}

		// Initialize
		virtual void init(const ClauseDB& db, const Vass& va, int msize) 
		{
			// Set right model_size
			DPLL_Selection::init(db,va,msize);
			// Clear counters
			counter.clear();
			// Counter has every value [-msize,msize] except 0
			for (int k = -msize; k < 0; ++k) counter.insert(counter_type::value_type(k,0.0));
			for (int k = msize; k > 0; --k) counter.insert(counter_type::value_type(k,0.0));
			// Create counter by traversing all variables
			for_each(db.begin(),db.end(),[&](const PClause& c){
				counter_type& cref = counter; // fix for bug in MSVS C++ 2010
				for_each(c.begin(),c.end(),[&](int l){++cref[l];});
			});
			// Traverse all unit clauses
			for_each(va.begin(),va.end(),[&](const pvar* a){ ++counter[a->val]; });
		}

		// Choice Point: Select a literal
		virtual int choose(const ClauseDB&, const Vass& va) 
		{
			//cerr << "Searching for unassigned variable in counter: " <<  counter.size() << "," << vass.size() << "\n";
			// First, create all variables
			set<int> uvars;
			for (int k = 1; k <= model_size; ++k) uvars.insert(k);
			// Now filter off variables that have been assigned
			for_each(va.begin(),va.end(),[&](const pvar* a){uvars.erase(std::abs(a->val));});
			// uvars now contains all unassigned variables, pick best
			//cerr << "Number of unassigned variables: " << uvars.size() << "\n";
			assert(!uvars.empty()); // can't be empty as we've already checked vass.size != model_size
			int lit = 0;
			double litcv = -numeric_limits<double>::max();
			// Get assignment with highest counter
			auto lit_less = [](int x, int y){ return x == -y ? x < 0 : abs(x) < abs(y); };
			for (auto v = uvars.begin(); v != uvars.end(); ++v) {
				// negative
				double vd = counter[-*v];
				if (vd > litcv || vd == litcv && lit_less(-*v,lit)) {
					lit = -*v;
					litcv = vd;
				}
				// positive (ignore if ones == max1)
				if (va.is_assignable(*v)) {
					vd = counter[*v];
					if (vd > litcv || vd == litcv && lit_less(*v,lit)) {
						lit = *v;
						litcv = vd;
					}
				}
			}
			return lit;
		}

		// Update
		virtual void update(const ClauseDB&, const Vass&, const PClause& constraint) 
		{
			if (!constraint.unit()) {
				// Before adding constraint, scale down all current counter values
				for_each(counter.begin(),counter.end(),[&](counter_type::value_type& p){p.second *= scale;});
				// Increase counter by 1.0 for literals of constraint
				for_each(constraint.begin(),constraint.end(),[&](int l){++counter[l];});
			}
		}
	protected:
		counter_type counter;
		double scale;
	};

	// Stochastic VSIDS
	class sVSIDS_selection : public VSIDS_selection {
	public:
		sVSIDS_selection(double s) : VSIDS_selection(s) {}
		// Override selection UPDATE: use override here
		virtual int choose(const ClauseDB&, const Vass& va) 
		{
			//cerr << "Searching for unassigned variable in counter: " <<  counter.size() << "," << vass.size() << "\n";
			// First, create all variables
			set<int> uvars; // both positive and negative assignments separately
			for (int k = -model_size; k < 0; ++k) uvars.insert(k);
			for (int k = 1; k <= model_size; ++k) uvars.insert(k);
			// Now filter off variables that have been assigned
			for_each(va.begin(),va.end(),[&](const pvar* a){ uvars.erase(a->val); uvars.erase(-a->val); });
			// uvars now contains all unassigned variables, pick best
			//cerr << "Number of unassigned variables: " << uvars.size() << "\n";
			assert(!uvars.empty()); // can't be empty as we've already checked vass.size != model_size
			// Compute absolute weight sum of variables that have NOT been assigned
			// Note: counts of 0 don't need to be assigned (they're value doesn't matter), so we just "pass them by"
			double wsum = 0.0;
			//cerr << "uvars:\n";
			//copy(uvars.begin(),uvars.end(),ostream_iterator<int>(cerr," "));
			//cerr << "counter:\n";
			//for_each(counter.begin(),counter.end(),[&](const counter_type::value_type& p){cerr << p.first << " -> " << p.second << "\n"; });
			//cerr << "wsum:\n";
			for_each(counter.begin(),counter.end(),[&](const counter_type::value_type& p){
				if (uvars.find(p.first) != uvars.end() && va.is_assignable(p.first)) {
					// cerr << "  " << p.first << " -> " << p.second << "\n";
					wsum += std::abs(p.second);
				}
			});
			// Get a random number between 0.0 and wsum
			const double pd = useful::uniform_continuous(0.0,wsum);
			// Pick assignment
			double d = 0;
			auto i = counter.begin();
			for ( ; i != counter.end(); ++i) {
				if (uvars.find(i->first) != uvars.end() && va.is_assignable(i->first)) {
					d += std::abs(i->second);
					if (d >= pd) break;
				}
			}
			return i->first;
		}
	};


	inline unique_ptr<DPLL_Selection> get_dpll_selection(const parameters& params)
	{
		const data& dat = params[parameters::dpll_strategy];
		if (dat == "auto") {
			return nullptr; // let algorithm choose
		} else if (dat == "vsids") {
			return unique_ptr<DPLL_Selection>(new VSIDS_selection(1.0/params.force_float(parameters::vsids_decay)));
		} else if (dat == "svsids") {
			return unique_ptr<DPLL_Selection>(new sVSIDS_selection(1.0/params.force_float(parameters::vsids_decay)));
		} else if (dat == "random") {
			return unique_ptr<DPLL_Selection>(new Random_selection());
		} else if (dat == "complexity") {
			return unique_ptr<DPLL_Selection>(new Simplicity_selection());
		} else {
			DEBUG_WARNING(cerr << "warning: invalid DPLL strategy \"" << dat << "\"\n");
			return nullptr;
		}
	}


	//============== Inlined functions ============//

	inline bitstring make_bitstring(const Vass& m)
	{
		bitstring bs(m.size(),0);
		for_each(m.begin(),m.end(),[&](const pvar* a){
			if (a->val > 0) bs[a->val-1] = 1;
		});
		return bs;
	}

	//inline bitstring make_bitstring(const term* bot, const term* cand)
	//{
	//	bitstring bs;
	//	auto citer = cand->body_begin();
	//	auto cend = cand->body_end();
	//	auto biter = bot->body_begin();
	//	auto bend = bot->body_end();
	//	for ( ; citer != cend; ++biter) {
	//		if (*biter == *citer) {
	//			bs.push_back(1);
	//			++citer;
	//		} else {
	//			bs.push_back(0);
	//		}
	//	}
	//	// Add zeros to tail
	//	bs.insert(bs.end(),distance(biter,bend),0);
	//	assert(citer == cand->body_end());
	//	assert(count(bs.begin(),bs.end(),1) == cand->body_size());
	//	return bs;
	//}

	inline PClause make_complement_clause(const Vass& va, int consistent)
	{
		//cerr << "make_compl_clause for: " << bs << ", consistent: " << consistent << "\n";
		PClause c;
		if (consistent < 0) {
			// Prune up 0110 => 0**0 => -b1 /\ -b4 => allowed: b1 \/ b4
			for_each(va.begin(),va.end(),[&](const pvar* a){ if (a->val < 0) c.insert(-a->val); });
		} else if (consistent > 0) {
			// Prune down 0110 => *11* => b2 /\ b3 => allowed: -b2 \/ -b3
			for_each(va.begin(),va.end(),[&](const pvar* a){ if (a->val > 0) c.insert(-a->val); });
		} else {
			// Prune one [0,1,1,0] => 0110 => -b1 /\ b2 /\ b3 /\ -b4 => b1 \/ -b2 \/ -b3 \/ b4
			for_each(va.begin(),va.end(),[&](const pvar* a){ c.insert(-a->val); });
		}
		//cerr << "make_compl_clause done: " << c << "\n";
		return c;
	}

	//inline term* make_functor_ptr(const term* bot, const bitstring& bs)
	//{
	//	term* f = new term[3];
	//	f[0] = term(if_id,2);
	//	f[1] = bot->head();
	//	unsigned k = 0;
	//	term* last = nullptr; // optimize body_push_back by storing last element
	//	std::for_each(bot->body_begin(),bot->body_end(),[&](const term& bl){
	//		if (bs[k] == 1) {
	//			term* el = bl.copy();
	//			f.body_push_back( el , last );
	//			last = el;
	//		}
	//		++k;
	//	});
	//	assert(k == bs.size());
	//	return f;
	//}

	//inline std::unique_ptr<Functor> make_functor(const term* bot, const bitstring& bs)
	//{
	//	return std::unique_ptr<Functor>(make_functor_ptr(&bot,bs));
	//}
	//
	//inline term* make_functor_ptr(const term* bot, const Vass& va)
	//{
	//	return make_functor_ptr(bot,make_bitstring(va));
	//}

	//inline std::unique_ptr<Functor> make_functor(const term* bot, const Vass& va)
	//{
	//	return std::unique_ptr<Functor>(make_functor_ptr(&bot,va));
	//}


	inline PClause::PClause(const bitstring& bs)
	{
		const int bsize = int(bs.size());
		for (int k = 0; k < bsize; ++k) {
			if (bs[k] == 1) insert(k+1);
		}
	}

	inline bool PClause::insert(int lit)
	{
		auto p = lits.insert(lit);
		if (!p.second) return false;
		if (watch.second == 0) {
			if (watch.first == 0) watch.first = lit;
			else watch.second = lit;
		} // else: don't change watch list
		return true;
	}

	inline bool PClause::erase(int lit)
	{
		auto s = lits.erase(lit);
		if (!s) return false;
		if (watch.first == lit) {
			// move watch.second down to watch.first
			watch.first = watch.second;
			watch.second = lit; // will trigger if below
		}
		if (watch.second == lit) {
			auto i = begin();
			if (i != end()) { // at least 1 elements
				if (*i != watch.first) {
					watch.second = *i;
				} else {
					if (++i != end()) { // 2 or more elements
						watch.second = *i;
					} else { // exactly 1 element
						watch.second = 0;
					}
				}
			} else { // came from previous if, empty clause now
				watch.second = 0;
				assert(watch.first == 0);
			}
		}
		return true;
	}

	inline int PClause::is_function(int l) const
	{
		if (lits.find(l) != lits.end()) return 1;
		else if (lits.find(-l) != lits.end()) return -1;
		else return 0;
	}

	inline int PClause::other_watched(int l) const
	{
		if (watch.first==l) return watch.second;
		else if (watch.second==l) return watch.first;
		else return 0;
	}

	inline bool PClause::switch_watched(int lo, int ln)
	{
		if (watch.first == lo) {
			watch.first = ln;
			return true;
		} else if (watch.second == lo) {
			watch.second = ln;
			return true;
		} else return false;
	}

	inline bool PClause::satisfied(const Vass& va) const
	{
		for (auto l = lits.begin(); l != lits.end(); ++l) {
			const pvar& pv = va.get_var(*l);
			if (pv.val == *l) return true;
		}
		return false;
	}


	inline void ClauseDB::insert(PClause&& c)
	{
		if (c.empty()) {
			// Empty clause is tagged individually
			empty_cl = true;
		} else if (c.unit()) {
			// If it's a unit clause, add assignment only
			ass.push_back(c.unit());
		} else {
			// Add to DB
			db.push_back(move(c));
		}
	}


	//inline bool Vass::operator==(const Vass& vass) const
	//{
	//	const size_type size1 = va.size();
	//	const size_type size2 = vass.va.size();
	//	for (size_type i = 0; i < size1; ++i) {
	//		size_type j = 0;
	//		for ( ; j < size2; ++j) {
	//			if (va[i] == vass.va[j]) break;
	//		}
	//		if (j == size2) return false;
	//	}
	//	return true;
	//}


	inline Vass& Vass::operator=(const Vass& v)
	{
		if (this != &v) {
			ones = v.ones;
			m1 = v.m1;
			ac = v.ac;
			amax = v.amax;
			pvt = v.pvt;
			for (const pvar* p : v.va) {
				// Which pvt does w point to?
				const int k = (&v.pvt.back() - p);
				va.push_back(&pvt[k]);
			}
			// Make new bit_to_count
			bit_to_count.clear();
			bit_to_count.reserve(v.bit_to_count.size());
			std::map<const pair<int,int>*,int> ptr_map;
			int k = 0;
			for_each(v.bit_to_count.begin(),v.bit_to_count.end(),[&](const shared_ptr<pair<int,int>>& p){
				auto at = ptr_map.find(p.get());
				if (at == ptr_map.end()) {
					// New
					ptr_map.insert(std::make_pair(p.get(),k));
					bit_to_count.push_back(shared_ptr<pair<int,int>>(new pair<int,int>(*p)));
				} else {
					// Old
					const int n = at->second;
					bit_to_count.push_back(bit_to_count[n]);
				}
				++k;
			});
			assert( count_if(va.begin(),va.end(),[](const pvar* a){return a->val > 0;}) == ones );
		}
		return *this;
	}

	inline Vass& Vass::operator=(Vass&& v) 
	{
		if (this != &v) {
			ones = v.ones;
			m1 = v.m1;
			ac = v.ac;
			amax = v.amax;
			pvt = std::move(v.pvt);
			va = std::move(v.va);
			bit_to_count = std::move(v.bit_to_count); // does not point into anything so move it
			assert( count_if(va.begin(),va.end(),[](const pvar* a){return a->val > 0;}) == ones );
		}
		return *this;
	}

	inline void Vass::clear_ass()
	{
		pvt.clear();
		va.clear(); 
		ac = 0;
		ones = 0;
	}

	inline void Vass::clear() 
	{
		clear_ass();
		m1 = numeric_limits<int>::max();
		amax = numeric_limits<int>::max();
		bit_to_count.clear();
	}

	inline bool Vass::is_assignable(int a) const 
	{
		if (a < 0) return true; // 0's can always be assigned
		// a is positive, check max1
		if (ones >= m1) return false; // no, we've reached max1
		// check recall (here: a > 0)
		const shared_ptr<pair<int,int>>& sptr = bit_to_count[a-1];
		return sptr->first < sptr->second; // recall not filled?
	}

	inline bool Vass::set_max1(int max) 
	{
		if (ones > max) return false;
		// Update m1
		m1 = max;
		assert( count_if(va.begin(),va.end(),[](const pvar* a){return a->val > 0;}) == ones );
		return true;
	}

	inline bool Vass::set_maxa(int max) 
	{
		if (ac > max) return false;
		// Update amax
		amax = max;
		assert( int(va.size()) <= amax );
		return true;
	}

	inline void Vass::reset(int model_size)
	{
		assert(bit_to_count.size() == model_size);
		reset_counters();
		init_model(model_size);
	}

	inline void Vass::reset_counters()
	{
		ac = 0; 
		for (const auto& sp : bit_to_count) {
			//std::cerr << "Bit_to_count: " << sp->first << " / " << sp->second << "\n";
			sp->first = 0;
		}
	}

}

#endif

